<?php
/**
 * Plugin Name: ACE WebP Optimizer
 * Description: Automatically converts images to WebP format for faster page loading.
 * Version: 1.0.0
 * Author: ACE
 * Requires PHP: 8.0
 * License: GPLv2 or later
 */

if (!defined('ABSPATH')) exit;

define('ACE_WEBP_VERSION', '1.0.0');
define('ACE_WEBP_PATH', plugin_dir_path(__FILE__));
define('ACE_WEBP_URL', plugin_dir_url(__FILE__));

require_once ACE_WEBP_PATH . 'includes/class-converter.php';
require_once ACE_WEBP_PATH . 'includes/class-admin.php';
require_once ACE_WEBP_PATH . 'includes/class-frontend.php';

// Set default options on activation.
register_activation_hook(__FILE__, function (): void {
    if (get_option('ace_webp_quality') === false) {
        add_option('ace_webp_quality', 82);
    }
    if (get_option('ace_webp_auto_convert') === false) {
        add_option('ace_webp_auto_convert', 1);
    }

    $converter = new ACE_WebP_Converter();
    if (!$converter->is_supported()) {
        set_transient('ace_webp_no_support', true, 300);
    }
});

// Admin notice when server lacks WebP support.
add_action('admin_notices', function (): void {
    if (get_transient('ace_webp_no_support')) {
        echo '<div class="notice notice-error"><p>';
        echo '<strong>ACE WebP Optimizer:</strong> Your server does not support WebP conversion. ';
        echo 'Please enable the GD library with WebP support or install the Imagick extension.';
        echo '</p></div>';
        delete_transient('ace_webp_no_support');
    }
});

// Initialize admin and frontend.
add_action('init', function (): void {
    $converter = new ACE_WebP_Converter();

    if (is_admin()) {
        new ACE_WebP_Admin($converter);
    } else {
        new ACE_WebP_Frontend();
    }
});

// Auto-convert images on upload.
add_filter('wp_generate_attachment_metadata', function (array $metadata, int $attachment_id): array {
    if (!get_option('ace_webp_auto_convert', 1)) {
        return $metadata;
    }

    $converter = new ACE_WebP_Converter();
    if (!$converter->is_supported()) {
        return $metadata;
    }

    $converter->convert_attachment($attachment_id, $metadata);
    return $metadata;
}, 10, 2);

// Clean up WebP files when an attachment is deleted.
add_action('delete_attachment', function (int $attachment_id): void {
    $file = get_attached_file($attachment_id);
    if (!$file) return;

    $webp = $file . '.webp';
    if (file_exists($webp)) {
        unlink($webp);
    }

    $metadata = wp_get_attachment_metadata($attachment_id);
    if (!empty($metadata['sizes'])) {
        $dir = dirname($file);
        foreach ($metadata['sizes'] as $size) {
            $thumb_webp = $dir . '/' . $size['file'] . '.webp';
            if (file_exists($thumb_webp)) {
                unlink($thumb_webp);
            }
        }
    }
});
